﻿using System.Collections;
using System.Collections.Generic;
using TMPro;
using UnityEngine;

//StreetFlyoutController manages the mission results overlay
public class MissionResultsOverlayController : MonoBehaviour
{
    //UI Elements
    public GameObject PassedUI;
    public GameObject FailedUI;
    public TMP_Text RewardText;
    public Animator Animator;

    public float ShowTime = 5000.0f;

    private bool _IsAnimating = false;
    private float _TimeAnimated = 0.0f;

    /// <summary>
    /// Shows the mission passed UI
    /// </summary>
    /// <param name="rewardText">The reward text to display</param>
    public void ShowPassed(string rewardText)
    {
        //Set up the UI
        RewardText.SetText(rewardText);
        PassedUI.SetActive(true);
        FailedUI.SetActive(false);

        //Handle the audio
        AudioManager.Instance.StopFile(GameManager.Instance.CurrentMusicFile);
        GameManager.Instance.CurrentMusicFile = AudioManager.Instance.LoadedFiles["MissionPassedSFX"];
        AudioManager.Instance.PlayFile(GameManager.Instance.CurrentMusicFile);

        //Now show the UI
        Show();
    }

    /// <summary>
    /// Shows the mission failed UI
    /// </summary>
    public void ShowFailed()
    {
        //Set up the UI, restart the music and show the UI
        PassedUI.SetActive(false);
        FailedUI.SetActive(true);
        GameManager.Instance.RestartAmbientMusic();
        Show();
    }

    /// <summary>
    /// Handler for when the hide animation completes
    /// </summary>
    public void OnHideAnimationCompleted()
    {
        //De-activate all the UI
        PassedUI.SetActive(false);
        FailedUI.SetActive(false);
    }

    /// <summary>
    /// Shows the UI
    /// </summary>
    private void Show()
    {
        //Reset our animation variables and begin animating
        _TimeAnimated = 0.0f;
        _IsAnimating = true;
        Animator.Play("Show");
    }

    /// <summary>
    /// Hides the UI
    /// </summary>
    private void Hide()
    {
        //Reset our animation variables and begin animating
        _TimeAnimated = 0.0f;
        _IsAnimating = false;
        Animator.Play("Hide");
    }

    private void Start()
    {
        AudioManager.Instance.MusicFinished += OnMusicFinished;
    }

    /// <summary>
    /// Handler for the music finishing
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    private void OnMusicFinished(object sender, AudioManager.MusicFinishedEventArgs e)
    {
        if(e.Clip == AudioManager.Instance.LoadedFiles["MissionPassedSFX"].Clip)
        {
            //All finished with mission passed, restart the ambient
            GameManager.Instance.RestartAmbientMusic();
        }
    }

    private void Update()
    {
        if(_IsAnimating)
        {
            //If we're animating, increment the time
            _TimeAnimated += (Time.deltaTime * 1000.0f);

            if(_TimeAnimated >= ShowTime)
            {
                //Time up, let's hide
                Hide();
            }
        }
    }
}
